// debug.cpp starts
// CGP, 8/29/00
/*
 * Changes:
 *	CGP, 1/9/01,
 *		Changed to be a class so that we can have a debug thread running to enable
 *		one state value for the number of debug messages output. 
 */

/*
	errorType == 'n'; "normal" (only in verbose output)
	errorType == 'e'; "error" (in verbose and major outout)
	errorType == 't'; "temporary" (in verbose and major output)
*/

#include "Debug.h"
#include <stdio.h>
#include <string.h>
#include "DebugClient.h"

#define	ERROR_HEADER		"         ***!!! ERROR !!!***: "
#define	BUFFER_SIZE 		256


// See RatWindow.cpp for globals
extern DebugServer *GLOB_debug;

void DebugGUI(const char *s1)
{
	(new BAlert("Debug", s1, "Continue"))->Go();
}

void DebugBoot(const char *msg)
{
	SendDebugMonitor('e', msg);
}

void DebugBoot(char errorType, const char *msg1, const char *msg2)
{
	SendDebugMonitor(errorType, msg1, msg2);
}

long Dispatch(void *debugObject) // parameter is of type Debug *
{
	int8 errorType;
	BString message1, message2;
	
	DebugServer *obj = (DebugServer *) debugObject;
	
	for (;;) {
		// grab the next incoming message from the port queue
		BMessage *msg = obj->port->GetMessage();
		
		switch (msg->what) {
		case DEBUG_CHECK:
			obj->CheckDebug();
			break;
		
		case DEBUG_DEBUG_MSG:
			msg->FindInt8("errorType", &errorType);
			msg->FindString("message1", &message1);
			msg->FindString("message2", &message2);
			obj->DebugMsg((char) errorType, message1.String(), message2.String());
			break;
			
		default:
			DebugBoot("Debug::Dispatch: Unexpected message received");
			break;
		}
		
		delete msg;
	}
	
	return 0;
}

DebugServer::DebugServer()
{
	numErrors = 0;
	port = new PortMessage("DebugServer"); // need this before spawning thread
	
	// Start a thread to receive messages. First, make the thread.
	debugThreadID = spawn_thread(Dispatch,
		"receive messages", B_NORMAL_PRIORITY, (void *) this);
	if (debugThreadID < 0) {
		// failure
		DebugBoot("Debug::Debug: Error: Failed to start Dispatch thread!");
    }
    
    // Now, start the thread.
    if (resume_thread(debugThreadID) != B_OK) {
    	DebugBoot("Debug::Debug: Error: Could not resume_thread");
    }
}

DebugServer::~DebugServer()
{
	delete port;
	// end the thread
}

void DebugServer::SendMessage(BMessage *msg)
{
	port->SendMessage(msg);
}

void DebugServer::DebugMsg(char errorType, const char *m1, const char *m2)
{
	SendDebugMonitor(errorType, m1, m2);
	
	if (errorType == 'e') {
		numErrors++;
	}
}

void SendDebugMonitor(char errorType, const char *s1, const char *s2)
{
	char buffer[BUFFER_SIZE];

	const char *m1=s1, *m2=s2;

	if (s1 == NULL) m1 = "";
	if (s2 == NULL) m2 = "";

	if ((strlen(m1) + strlen(m2) + 1 + sizeof(ERROR_HEADER)) > BUFFER_SIZE) {
		strcat(buffer, "Debug message too long");
	} else {
		buffer[0] = '\0';
		if (errorType == 'e') {
			strcat(buffer, ERROR_HEADER);
		}
		strcat(buffer, m1);
		strcat(buffer, m2);
	}

	if (errorType == 'n') {
		tracex("Debug", dbgVERBOSE, buffer);
	} else {
		tracex("Debug", dbgMAJOR, buffer);
	}
}

void DebugServer::CheckDebug()
{
	char buffer[BUFFER_SIZE];
	
	if (numErrors > 0) {
		sprintf(buffer, "Warning: %d errors have occurred in debug output.", numErrors);
		numErrors = 0;
		(new BAlert("Debug", buffer, "Continue"))->Go();
		return;
	}
}

void Debug(const char *message1, const char *message2)
{
	Debug('t', message1, message2);
}

void CheckDebug()
{
	// Send a message to the debug thread
	BMessage msg;
	
	msg.what = DEBUG_CHECK;
	GLOB_debug->SendMessage(&msg); // send port message to debug thread
}

void Debug(char errorType, const char *s1, const char *s2)
{
	const char *m1=s1, *m2=s2;

	if (s1 == NULL) m1 = "";
	if (s2 == NULL) m2 = "";
	
	// Send a message to the debug thread
	BMessage msg;
	msg.what = DEBUG_DEBUG_MSG;
	
	msg.AddInt8("errorType", (int8) errorType);
	msg.AddString("message1", m1);
	msg.AddString("message2", m2);
	GLOB_debug->SendMessage(&msg);
}

// debug.cpp ends